<?php
declare(strict_types=1);

/**
 * FINAL — composerless build, with fixes:
 * - Robust panel response parsing (array/object, field 'link' fallback)
 * - "pending" lock to prevent duplicate trial issuance on repeated clicks
 * - Schema auto-migration for trials.status ENUM to include 'pending'
 * - Browser helpers for webhook set/delete/info
 */

// === Telegram & Admin ===
define('BOT_TOKEN', '7786762420:AAF0tGUG-p8-0FjP8iQ1WFccSk1g6TKHexw');
define('ADMIN_ID', 485974008);
define('CHANNEL_ID', '@dezhnets');

// === Database ===
define('MYSQL_HOST', 'localhost');
define('MYSQL_DB',   'cgnplftk_vpn');
define('MYSQL_USER', 'cgnplftk_vpn');
define('MYSQL_PASS', 'BFyLX,c6OeJx');

// === Panel ===
define('PANEL_BASE', 'https://api.adspeedo.com');
// IMPORTANT: replace below with your actual panel token (WITHOUT the "Bearer " prefix)
define('PANEL_TOKEN', ''); // ← fill this with the real token string

// === Misc ===
date_default_timezone_set('Asia/Tehran');
ini_set('default_charset', 'UTF-8');

// --- DB (PDO) ---
function db(): PDO {
    static $pdo = null;
    if ($pdo === null) {
        $dsn = 'mysql:host=' . MYSQL_HOST . ';dbname=' . MYSQL_DB . ';charset=utf8mb4';
        $pdo = new PDO($dsn, MYSQL_USER, MYSQL_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]);
    }
    return $pdo;
}

function init_schema(): void {
    $pdo = db();

    // Create tables if missing
    $pdo->exec("
    CREATE TABLE IF NOT EXISTS users (
      id INT AUTO_INCREMENT PRIMARY KEY,
      chat_id BIGINT UNIQUE,
      username VARCHAR(64),
      first_name VARCHAR(64),
      joined BOOLEAN DEFAULT 0,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    $pdo->exec("
    CREATE TABLE IF NOT EXISTS trials (
      id INT AUTO_INCREMENT PRIMARY KEY,
      chat_id BIGINT UNIQUE,
      panel_service_id VARCHAR(64),
      panel_service_uuid VARCHAR(128),
      subscription_link TEXT,
      status ENUM('pending','success','failed','expired') DEFAULT 'pending',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // Migrate existing tables to ensure 'pending' state exists
    try {
        $pdo->exec("ALTER TABLE trials MODIFY status ENUM('pending','success','failed','expired') DEFAULT 'pending';");
    } catch (\Throwable $e) {
        // ignore if already up to date
    }
}

// --- Minimal Telegram HTTP client (no SDK) ---
function tg_api(string $method, array $params = []): array {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $method;
    $ch  = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $params,
        CURLOPT_TIMEOUT => 20,
    ]);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);
    if ($resp === false) {
        error_log("tg_api {$method} cURL error: {$err}");
        return ['ok' => false, 'description' => $err];
    }
    $data = json_decode($resp, true);
    if (!is_array($data)) {
        error_log("tg_api {$method} non-JSON: {$resp}");
        return ['ok' => false, 'description' => 'non-json'];
    }
    if (!($data['ok'] ?? false)) {
        error_log("tg_api {$method} failed: {$resp}");
    }
    return $data;
}

function send_text($chat_id, string $text): void {
    tg_api('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ]);
}

function send_menu($chat_id, string $text): void {
    $keyboard = [
        'inline_keyboard' => [[
            ['text' => '✅ تأیید عضویت', 'callback_data' => 'verify'],
            ['text' => '🎁 دریافت تست رایگان', 'callback_data' => 'trial'],
        ]]
    ];
    tg_api('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $text,
        'reply_markup' => json_encode($keyboard, JSON_UNESCAPED_UNICODE),
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ]);
}

function answer_cb($cb_id): void {
    tg_api('answerCallbackQuery', ['callback_query_id' => $cb_id]);
}

function check_membership($user_id): bool {
    $res = tg_api('getChatMember', [
        'chat_id' => CHANNEL_ID,
        'user_id' => $user_id
    ]);
    if (!($res['ok'] ?? false)) return false;
    $status = $res['result']['status'] ?? null;
    // Allow common "in channel" states
    return in_array($status, ['member','administrator','creator','restricted'], true);
}

// --- User & Trial helpers ---
function ensure_user($chat_id, $username, $first_name): void {
    $stmt = db()->prepare("INSERT INTO users (chat_id, username, first_name) VALUES (?, ?, ?)
                           ON DUPLICATE KEY UPDATE username=VALUES(username), first_name=VALUES(first_name)");
    $stmt->execute([$chat_id, $username, $first_name]);
}

function has_trial($chat_id): bool {
    $stmt = db()->prepare("SELECT id FROM trials WHERE chat_id=?");
    $stmt->execute([$chat_id]);
    return (bool)$stmt->fetchColumn();
}

function save_trial($chat_id, $service_id, $uuid, $subscription, $status): void {
    $stmt = db()->prepare("INSERT INTO trials (chat_id, panel_service_id, panel_service_uuid, subscription_link, status)
                           VALUES (?, ?, ?, ?, ?)
                           ON DUPLICATE KEY UPDATE panel_service_id=VALUES(panel_service_id),
                                                   panel_service_uuid=VALUES(panel_service_uuid),
                                                   subscription_link=VALUES(subscription_link),
                                                   status=VALUES(status)");
    $stmt->execute([$chat_id, $service_id, $uuid, $subscription, $status]);
}

// --- Text templates ---
function require_membership_text(): string {
    return "⚠️ لطفاً ابتدا در کانال ما عضو شوید:\n👉 " . CHANNEL_ID . "\nسپس روی دکمه «تأیید عضویت» کلیک کنید.";
}
function success_membership_text(): string {
    return "✅ عضویت شما تأیید شد. اکنون می‌توانید تست رایگان بگیرید.";
}
function already_trial_text(): string {
    return "⚠️ شما قبلاً تست رایگان دریافت کرده‌اید.";
}
function trial_created_text($link): string {
    $safe = htmlspecialchars((string)$link, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    return "🎉 اکانت تست شما ساخته شد!\n\n🔗 لینک اشتراک:\n{$safe}";
}
function panel_error_text(): string {
    return "❌ خطا در ارتباط با سرور. لطفاً بعداً دوباره تلاش کنید.";
}

// --- Panel API ---
function create_trial_on_panel($chat_id): array {
    if (!PANEL_TOKEN) {
        error_log("PANEL_TOKEN is empty — fill it in config.php");
        return [false, null];
    }

    $payload = [
        "user_id" => null,
        "package_id" => 136,
        "count" => 1,
        "batch" => false,
        "mobile" => "",
        "name" => "trial_for_" . $chat_id,
        "description" => "Free Trial by Bot",
        "servicewithoutphone" => true,
        "day" => 1,
        "traffic" => 1
    ];
    $url = rtrim(PANEL_BASE, '/') . '/services';

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "Authorization: Bearer " . PANEL_TOKEN,
        ],
        CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
        CURLOPT_TIMEOUT => 25,
    ]);
    $resp = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($resp === false || $http < 200 || $http >= 300) {
        error_log("Panel error: HTTP $http, err=$err, resp=$resp");
        return [false, null];
    }

    $data = json_decode($resp, true);
    if (!is_array($data)) {
        error_log("Panel response not JSON: $resp");
        return [false, null];
    }

    // Handle array-wrapped response: [{...}]
    if (isset($data[0]) && is_array($data[0])) {
        $data = $data[0];
    }
    // Handle {"data": {...}} or {"data": [{...}]}
    if (isset($data['data'])) {
        if (is_array($data['data']) && isset($data['data'][0])) {
            $data = $data['data'][0];
        } elseif (is_array($data['data'])) {
            $data = $data['data'];
        }
    }

    $id  = $data['id'] ?? ($data['service_id'] ?? null);
    $uuid = $data['uuid'] ?? null;

    // Map subscription link across possible keys
    $subscription = null;
    foreach (['subscription','subscription_link','link','subscriptionLink','subscription_url','subscriptionUrl'] as $k) {
        if (!empty($data[$k])) { $subscription = $data[$k]; break; }
    }

    if (!$id || !$uuid || !$subscription) {
        error_log("Panel missing fields: " . json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
        return [false, $data];
    }

    return [true, ['id' => (string)$id, 'uuid' => (string)$uuid, 'subscription' => (string)$subscription]];
}
